// RFID reader with Interstudy ATmegal28 board
// Reader pins : Enable - PD3, Serout - PD2 (RXD1)
// Reads and displays ID

// Includes 
#include <avr/io.h> 
#include <util\delay.h> 
#include "lcd.h"
#include "pin.h"

// Configuration
#define RFID_BAUDRATE     2400
#define RFID_BAUD_VALUE   (((F_CPU / (RFID_BAUDRATE * 16UL))) - 1)
#define RFID_ENABLE       PORTPIN(D, 3)
#define RFID_RX           PORTPIN(D, 2)
#define LED_GREEN         PORTPIN(C, 3)
#define LED_DEBUG         PORTPIN(B, 7)

//
// Display initialization
//
void display_init(void)
{
	lcd_init(LCD_DISP_ON);
	lcd_clrscr();
	lcd_puts("  RFID reader\n waiting for ID");
}

//
// Write ID on display
//
void display_write_id(char *id)
{
	lcd_gotoxy(0, 0);
	lcd_puts("ID: "); 
	lcd_puts(id);
}

//
// UART configuring
//
void uart_setup()
{
	// Setup serial interface
	SET_BIT(UCSR1B, RXEN);   // Activate RX only
	SET_BIT(UCSR1C, UCSZ0);  // 8 data bits, 1 stop bit, no parity
	SET_BIT(UCSR1C, UCSZ1);

	// Set baud rate
	UBRR1L = (RFID_BAUD_VALUE & 0xFF);
	UBRR1H = (RFID_BAUD_VALUE >> 8);
}

//
// Wait for UART incoming data
//
void uart_wait_rx()
{
	while (!IS_BIT_SET(UCSR1A, RXC)) { }
}

//
// Read UART data
//
inline unsigned char uart_read()
{
	return UDR1;
}

//
// RFID interface initialization
//
void rfid_init(void)
{
	// Setup UART
	uart_setup();

	// Setup enable and RX pin
	pin_setup_output(RFID_ENABLE); 		
	pin_setup_input(RFID_RX);
}

//
// RFID ID reading
//
void rfid_read_id(char *id)
{
	enum States { BEGIN, DATA, END } state = BEGIN;
	unsigned char data;
	unsigned char digits = 0;

	// Enable RFID with low signal
	pin_clear(RFID_ENABLE);

	// Cycle until tag ID received
	while (1)
	{	 
 		// Wait for data
		uart_wait_rx();		
		data = uart_read();		

		// Toggle debug indicator
		pin_toggle(LED_DEBUG);

		// What's present state and what's next ? 
		switch (state)
		{
			// Begin state - we expect start byte (0x0A)
			case BEGIN:
				// Got the start ?
				if (data == 0x0A)
				{
					state = DATA;
					
					// Disable RFID
					pin_set(RFID_ENABLE);
				}
				break;

			// Data state - we expect 10 bytes of ID
			case DATA:
				// Fill ID string
				id[digits++] = data;				

				// All digits arrived ?
				if (digits == 10) state = END;
				break; 

			// End state - we expect end byte (0x0D)
			case END:
				// Got the end ?
				if (data == 0x0D)
				{
					// Terminate the string
					id[digits] = '\0'; 

					// All done - return
					return;					
				}
				// Any other case - restart
				else
				{
					state = BEGIN;

					// Enable RFID with low signal
					pin_clear(RFID_ENABLE);
				}
				break; 			
		}
	}
}

//
// Main function  
//
int main(void)
{
	char id[11];

	// Initialization
	display_init();
	rfid_init();

	// Setup indicator pins
	pin_setup_output(LED_GREEN); 
	pin_setup_output(LED_DEBUG);
		
 	// Endless loop
	while (1)
	{		
		// Turn off green LED
		pin_set(LED_GREEN);

		// Read RFID tag ID
		rfid_read_id(id);

		// Light green LED
		pin_clear(LED_GREEN);

		// Display ID	
		display_write_id(id);		
	}
} 
